let currentEV = 10010;
const events = {};
let currentEventId = null;
let currentEvNumber = null;
const units = {};

let hasPriority = false;
const dispatcherId = "dispatcher_" + Math.floor(Math.random() * 10000); // Temporary random ID

document.getElementById("priorityToggle").onclick = () => {
  hasPriority = !hasPriority;
  const btn = document.getElementById("priorityToggle");
  btn.textContent = hasPriority ? "Priority Mode: ON" : "Enable Priority Mode";
  btn.style.backgroundColor = hasPriority ? "red" : "";
  btn.style.color = hasPriority ? "white" : "";
  console.log("Priority Mode is now", hasPriority ? "ENABLED" : "DISABLED");
};

function getTimestamp() {
  return new Intl.DateTimeFormat('en-NZ', {
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit',
    hour12: false,
    timeZone: 'Pacific/Auckland'
  }).format(new Date());
}

document.getElementById("addEventBtn").onclick = () => {
  const location = document.getElementById("locationInput").value.trim();
  const code = document.getElementById("codeInput").value.trim();
  const priority = document.getElementById("priorityInput").value.trim();

  if (!location || !code || !priority) {
    alert("Please fill out all event details.");
    return;
  }

fetch('php/add_event.php', {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify({ location, code, priority, dispatcherId, priorityFlag: hasPriority })
    })
  .then(res => res.text())
  .then(text => {
    console.log("RAW RESPONSE:", text); 
    const data = JSON.parse(text);
    if (data.success) {
      const event = data.event;
      events[event.evNumber] = event;
      addEventToTable(event);
      clearForm();
    } else {
      alert("Failed to add event.");
    }
  })
  .catch(err => {
    console.error("Fetch error:", err);
  });
};

function addEventToTable(event) {
  const tbody = document.querySelector(".event-table tbody");
  const row = document.createElement("tr");
  row.innerHTML = `
    <td>${event.priority}</td>
    <td>${event.code}</td>
    <td><button onclick="loadEvent('${event.evNumber}')">${event.evNumber}</button></td>
    <td>${event.location}</td>
    <td><button onclick="closeEvent(this, '${event.evNumber}')">✖</button></td>
  `;
  tbody.appendChild(row);
}

function loadEvent(evNumber) {
  const event = events[evNumber];
  if (!event) return;

  currentEvNumber = evNumber;
  currentEventId = parseInt(evNumber.replace("P", ""));
  currentEventId = evNumber;

  document.getElementById("locationInput").value = event.location;
  document.getElementById("codeInput").value = event.code;
  document.getElementById("priorityInput").value = event.priority;
  document.getElementById("evNumberInput").value = evNumber;

  document.querySelectorAll(".event-table tbody tr").forEach(tr => {
    tr.classList.remove("active-row");
  });
  const activeRow = [...document.querySelectorAll(".event-table tbody button")]
    .find(btn => btn.textContent === evNumber)?.closest("tr");
  if (activeRow) activeRow.classList.add("active-row");

  renderNotes(event.notes);
}

function renderNotes(notes) {
  const noteList = document.getElementById("noteList");
  noteList.innerHTML = "";

  const evNumber = document.getElementById("evNumberInput").value;

  for (let i = notes.length - 1; i >= 0; i--) {
    const li = document.createElement("li");
    li.innerHTML = `
      ${notes[i]}
      <button onclick="deleteNote('${evNumber}', ${i})">✖</button>
    `;
    noteList.appendChild(li);
  }
}
document.getElementById("addNoteBtn").onclick = () => {
  const noteText = document.getElementById("noteInput").value.trim();
  const evNumber = document.getElementById("evNumberInput").value;
  if (!noteText || !events[evNumber]) return;

  const timestamped = `[${getTimestamp()}] ${noteText}`;

fetch('php/add_note.php', {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify({ evNumber, note: timestamped, dispatcherId})
})
  .then(res => res.json())
  .then(data => {
    if (data.success) {
      events[evNumber].notes = data.notes;
      renderNotes(data.notes);
      document.getElementById("noteInput").value = "";
    } else {
      alert("Failed to add note.");
      console.error(data);
    }
  })
  .catch(err => {
    console.error("Add note fetch error:", err);
  });
};
function deleteNote(evNumber, index) {
fetch('php/delete_note.php', {
  method: 'POST',
  headers: { 'Content-Type': 'application/json' },
  body: JSON.stringify({ evNumber, index, dispatcherId})
})
  .then(res => res.json())
  .then(data => {
    if (data.success) {
      events[evNumber].notes = data.notes;
      renderNotes(data.notes);
    }
  });
}

document.getElementById("saveChangesBtn").onclick = () => {
  if (!currentEventId) return;

  const location = document.getElementById("locationInput").value.trim();
  const code = document.getElementById("codeInput").value.trim();
  const priority = document.getElementById("priorityInput").value.trim();

  events[currentEventId] = {
    ...events[currentEventId],
    location,
    code,
    priority
  };

  refreshEventTable();
  saveToLocal();
  clearForm();
  currentEventId = null;
};

document.getElementById("cancelChangesBtn").onclick = () => {
  const evNumber = document.getElementById("evNumberInput").value;
  if (events[evNumber]) {
    loadEvent(evNumber);
  } else {
    clearForm();
  }
};

function clearForm() {
  document.getElementById("locationInput").value = "";
  document.getElementById("codeInput").value = "";
  document.getElementById("priorityInput").value = "";
  document.getElementById("evNumberInput").value = "";
  document.getElementById("noteInput").value = "";
  document.getElementById("noteList").innerHTML = "";
  document.querySelectorAll(".event-table tbody tr").forEach(tr => tr.classList.remove("active-row"));
}

function closeEvent(btn, evNumber) {
fetch('php/delete_event.php', {
    method: 'POST',
    headers: {
        'Content-Type': 'application/json'
    },
    body: JSON.stringify({
        evNumber: '1234',
        dispatcherId: 'disp01',
        priority: true
    })
})
.then(res => res.json())
.then(data => {
    if (data.success) {
        alert("Job deleted!");
        location.reload();
    } else {
        alert("Failed to delete job: " + data.error);
    }
});

}

function refreshEventTable() {
  const tbody = document.querySelector(".event-table tbody");
  tbody.innerHTML = "";
  Object.values(events).forEach(addEventToTable);
}

const selectedUnits = new Set();

function clearUnitSelection() {
  selectedUnits.forEach(unit => unit.classList.remove("selected"));
  selectedUnits.clear();
  document.getElementById("statusBox").style.display = "none";
}

function addUnit() {
  const callsign = document.getElementById("unit-callsign").value.trim();
  const status = document.getElementById("unit-status").value;
  const evNumber = document.getElementById("unit-evnum").value.trim();
  const location = document.getElementById("unit-location").value.trim();

  if (!callsign) {
    alert("Callsign is required");
    return;
  }

  units[callsign] = { callsign, status, evNumber, location };

  fetch('php/add_unit.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      callsign,
      status,
      evNumber,
      location
    })
  })
  .then(res => res.json())
  .then(data => {
    if (!data.success) {
      alert("Unit sync failed: " + (data.error || "Unknown error"));
      console.error("Add unit error:", data);
    }
  })
  .catch(err => {
    console.error("Fetch error while adding unit:", err);
  });

  renderUnit(units[callsign]);
  document.getElementById("unit-callsign").value = "";
  document.getElementById("unit-evnum").value = "";
  document.getElementById("unit-location").value = "";
  saveToLocal();
}

document.addEventListener("DOMContentLoaded", () => {
  window.addUnit = addUnit;

  document.getElementById("closeJobBtn").addEventListener("click", () => {
    if (!currentEvNumber || !events[currentEvNumber]) return;
    document.getElementById("closeJobPopup").classList.remove("hidden");
  });

  document.getElementById("cancelCloseJob").addEventListener("click", () => {
    document.getElementById("closeJobPopup").classList.add("hidden");
  });

document.getElementById("confirmCloseJob").addEventListener("click", () => {
  const kCode = document.getElementById("kCodeSelect").value;
  const resultNote = `[${getTimestamp()}] Job closed with result: ${kCode}`;

  // Add note locally and re-render
  if (events[currentEvNumber]) {
    events[currentEvNumber].notes.push(resultNote);
    renderNotes(events[currentEvNumber].notes);
  }

  // Save note to server
  fetch('php/add_note.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      evNumber: currentEvNumber,
      note: resultNote
    })
  });

  // Delete event on server
  fetch('php/delete_event.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      evNumber: currentEvNumber,
      dispatcherId: "system",
      priority: true
    })
  })
  .then(res => res.json())
  .then(data => {
    if (data.success) {
      delete events[currentEvNumber];
      currentEvNumber = null;
      currentEventId = null;
      document.getElementById("evNumberInput").value = "";

      refreshData(); // 
      document.getElementById("closeJobPopup").classList.add("hidden"); // 
    } else {
      alert("Failed to delete job: " + (data.error || "Unknown error"));
    }
  })
})
.catch(err => {
  console.error("Delete event error:", err);
  alert("An error occurred while deleting the job.");
});
});

function renderUnit(unitData) {
  const { callsign, status, evNumber, location } = unitData;
  const unitList = document.getElementById("unitList");

const existing = document.querySelector(`.unit-item[data-callsign="${unitData.callsign}"]`);
if (existing) existing.remove();

const div = document.createElement("div");
div.className = `unit-item status-${status.replace("/", "-")}`;
div.dataset.callsign = callsign;

const isLockedByOther = unitData.dispatcher && unitData.dispatcher !== dispatcherId;

  div.innerHTML = `
  <div>${callsign}</div>
  <div class="status">${status}</div>
  <div class="ev">${evNumber}</div>
  <div class="loc">${location}</div>
  ${isLockedByOther ? `<div class="locked">Locked by ${unitData.dispatcher}</div>` : ""}
  <div><button onclick="removeUnit(this)">✖</button></div>
`;
if (isLockedByOther) {
  div.style.opacity = "0.5";
  div.style.pointerEvents = "none"; // Prevent interaction
}

  div.onclick = function (e) {
    e.stopPropagation();
    if (e.target.tagName === "BUTTON") return;

    if (selectedUnits.has(div)) {
      selectedUnits.delete(div);
      div.classList.remove("selected");
    } else {
      selectedUnits.add(div);
      div.classList.add("selected");
    }

    if (selectedUnits.size > 0) {
      positionStatusBox(div);
    } else {
      document.getElementById("statusBox").style.display = "none";
    }
  };

  unitList.appendChild(div);
}

function removeUnit(btn) {
  const div = btn.closest(".unit-item");
  const callsign = div.dataset.callsign;

  fetch('php/delete_unit.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      callsign,
      dispatcherId,
//    // priority: hasPriority
    })
  })
  .then(res => res.json())
  .then(data => {
    if (data.success) {
      selectedUnits.delete(div);
      div.remove();
    } else {
      alert(data.error || "Failed to delete unit.");
      console.error(data);
    }
  });
}

function changeSelectedStatus(action) {
  const currentEvNumber = document.getElementById("evNumberInput").value.trim();
  if (action === "ASSIGN") {
    if (!currentEvNumber || !events[currentEvNumber]) {
      alert("No active event is selected to assign units.");
      return;
    }
  
    const assignedCallsigns = [];
  
    selectedUnits.forEach(unit => {
      const callsign = unit.dataset.callsign;
      const evDiv = unit.querySelector(".ev");
      const locDiv = unit.querySelector(".loc");
  
      evDiv.textContent = currentEvNumber;
      const location = events[currentEvNumber].location || "";
      locDiv.textContent = location;
  
      if (units[callsign]) {
        units[callsign].evNumber = currentEvNumber;
        units[callsign].location = location;
  
        fetch('php/add_unit.php', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            callsign,
            status: units[callsign].status,
            evNumber: currentEvNumber,
            location: location
          })
        })
        .then(res => res.json())
        .then(data => {
          if (!data.success) {
            alert("Unit sync failed: " + (data.error || "Unknown error"));
            console.error("Add unit error:", data);
          }
        })
        .catch(err => {
          console.error("Fetch error while assigning unit:", err);
        });
  
        assignedCallsigns.push(callsign);
      }
    });
  
    clearUnitSelection();
  
    const assignNote = `[${getTimestamp()}] Assigned units to ${currentEvNumber}: ${assignedCallsigns.join(", ")}`;
    events[currentEvNumber].notes.push(assignNote);
    renderNotes(events[currentEvNumber].notes);
  }
  
  else if (["10-2", "10-3", "10-7", "10-0", "10-5"].includes(action)) {
    const groupedCallsigns = [];
  
    selectedUnits.forEach(unit => {
      const callsign = unit.dataset.callsign;
      const statusDiv = unit.querySelector(".status");
      const evNumber = unit.querySelector(".ev").textContent;
  
      // Update UI
      unit.className = unit.className.replace(/status-[\w-]+/, `status-${action}`);
      statusDiv.textContent = action;
  
      // Update local object
      if (units[callsign]) {
        units[callsign].status = action;
  
        // Save updated status to DB
        fetch('php/add_unit.php', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            callsign,
            status: action,
            evNumber: units[callsign].evNumber || "",
            location: units[callsign].location || ""
          })
        })
        .then(res => res.json())
        .then(data => {
          if (!data.success) {
            alert(`Status update failed for ${callsign}: ` + (data.error || "Unknown error"));
            console.error("Update error:", data);
          }
        })
        .catch(err => {
          console.error("Fetch error for unit status update:", err);
        });
      }
  
      // Log notes for responding or on-scene
      if ((action === "10-2" || action === "10-7") && evNumber && events[evNumber]) {
        groupedCallsigns.push(callsign);
      } else if (evNumber && events[evNumber]) {
        const statusNote = `[${getTimestamp()}] ${callsign} status changed to ${action}`;
        events[evNumber].notes.push(statusNote);
        renderNotes(events[evNumber].notes);
      }
  
      unit.classList.remove("selected");
    });
  
    clearUnitSelection();
  
    if (groupedCallsigns.length > 0) {
      const evNumber = document.getElementById("evNumberInput").value;
      const groupNote = `[${getTimestamp()}] ${groupedCallsigns.join(", ")} status changed to ${action}`;
      events[evNumber]?.notes.push(groupNote);
      renderNotes(events[evNumber]?.notes || []);
    }
  
    saveToLocal();
  }

  else if (action === "CLEAR") {
    const clearedCallsigns = [];
  
    selectedUnits.forEach(unit => {
      const callsign = unit.dataset.callsign;
      const statusDiv = unit.querySelector(".status");
      const evDiv = unit.querySelector(".ev");
      const locDiv = unit.querySelector(".loc");
      const evNumber = evDiv.textContent;
  
      if (evNumber && events[evNumber]) {
        clearedCallsigns.push(callsign);
      }
  
      // Clear visual unit info
      evDiv.textContent = "";
      locDiv.textContent = "";
      unit.className = `unit-item status-10-3`;
      statusDiv.textContent = "10-3";
  
      // Clear JS object and sync with backend
      if (units[callsign]) {
        units[callsign].status = "10-3";
        units[callsign].evNumber = "";
        units[callsign].location = "";
  
        fetch('php/add_unit.php', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            callsign,
            status: "10-3",
            evNumber: "",
            location: ""
          })
        })
        .then(res => res.json())
        .then(data => {
          if (!data.success) {
            alert("Unit CLEAR failed: " + (data.error || "Unknown error"));
            console.error("Clear error for", callsign, data);
          }
        })
        .catch(err => {
          console.error("Fetch error while clearing unit:", err);
        });
      }
  
      unit.classList.remove("selected");
    });
  
    // Add note to event if units were cleared
    if (clearedCallsigns.length > 0) {
      const evNumber = document.getElementById("evNumberInput").value;
      const clearNote = `[${getTimestamp()}] Cleared units from ${evNumber}: ${clearedCallsigns.join(", ")}`;
      events[evNumber]?.notes.push(clearNote);
      renderNotes(events[evNumber]?.notes || []);
    }
  
    clearUnitSelection();
    saveToLocal();
  }}

function positionStatusBox(target) {
  const rect = target.getBoundingClientRect();
  const box = document.getElementById("statusBox");
  const boxWidth = 150;
  const boxHeight = 200;

  let left = rect.right + 10;
  let top = rect.top + window.scrollY;

  if (left + boxWidth > window.innerWidth) {
    left = rect.left - boxWidth - 10;
  }
  if (top + boxHeight > window.innerHeight + window.scrollY) {
    top = window.innerHeight + window.scrollY - boxHeight - 10;
    if (top < 0) top = 10;
  }

  box.style.left = `${left}px`;
  box.style.top = `${top}px`;
  box.style.display = "block";
}

function updateNZTime() {
  const options = {
    timeZone: 'Pacific/Auckland',
    hour12: false,
    hour: '2-digit',
    minute: '2-digit',
    second: '2-digit'
  };

  const formatter = new Intl.DateTimeFormat('en-NZ', options);
  const now = new Date();
  document.getElementById('nzClock').textContent = 'NZ Time: ' + formatter.format(now);
}

setInterval(updateNZTime, 1000);
updateNZTime();

document.addEventListener("click", (e) => {
  const box = document.getElementById("statusBox");
  if (!box.contains(e.target) && ![...selectedUnits].some(unit => unit.contains(e.target))) {
    selectedUnits.forEach(unit => unit.classList.remove("selected"));
    selectedUnits.clear();
    box.style.display = "none";
  }
});

function refreshData() {
  console.log("Checking refresh pause conditions...");

  if (currentEventId) {
    console.log("🔁 Skipping refresh: currentEventId is active.");
    return;
  }

  if (selectedUnits && selectedUnits.size > 0) {
    console.log("🔁 Skipping refresh: unit(s) selected.", Array.from(selectedUnits));
    return;
  }

  console.log("✅ Performing refresh.");
  // Refresh event list
  fetch('php/get_events.php')
    .then(res => res.json())
    .then(data => {
      // Clear current events
      Object.keys(events).forEach(key => delete events[key]);
      document.querySelector(".event-table tbody").innerHTML = "";

      data.forEach(event => {
        events[event.evNumber] = event;
        addEventToTable(event);
      });

      // If viewing selected event, reload it
      const current = document.getElementById("evNumberInput").value;
      if (events[current]) loadEvent(current);
    });

  // Refresh unit list
  fetch('php/api_units.php')
    .then(res => res.json())
    .then(data => {
      Object.keys(units).forEach(key => {
        const el = document.querySelector(`.unit-item[data-callsign="${key}"]`);
        if (el) el.remove();
      });
      Object.keys(units).forEach(key => delete units[key]);

      Object.assign(units, data);
      Object.values(units).forEach(renderUnit);
    });
}

document.getElementById("closeJobBtn").addEventListener("click", () => {
  console.log("Close Job button clicked!");
  if (!currentEventId || !events[currentEventId]) {
    console.log("No current event selected.");
    return;
  }
  console.log("Opening popup!");
  document.getElementById("closeJobPopup").classList.remove("hidden");
});

document.getElementById("cancelCloseJob").addEventListener("click", () => {
  document.getElementById("closeJobPopup").classList.add("hidden");
});

document.getElementById("confirmCloseJob").addEventListener("click", () => {
  const kCode = document.getElementById("kCodeSelect").value;
  const resultNote = `[${getTimestamp()}] Job closed with result: ${kCode}`;

  events[currentEventId].notes.push(resultNote);
  renderNotes(events[currentEventId].notes);

  fetch('php/add_note.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      evNumber: currentEventId,
      note: resultNote
    })
  });

  document.getElementById("closeJobPopup").classList.add("hidden");
});

function saveToLocal() {
  localStorage.setItem("cadEvents", JSON.stringify(events));
  localStorage.setItem("cadUnits", JSON.stringify(units));
}

function loadFromLocal() {
  const storedEvents = localStorage.getItem("cadEvents");
  const storedUnits = localStorage.getItem("cadUnits");

  if (storedEvents) {
    Object.assign(events, JSON.parse(storedEvents));
    refreshEventTable();
  }

  if (storedUnits) {
    Object.assign(units, JSON.parse(storedUnits));
    Object.values(units).forEach(renderUnit);
  }
}

window.onload = () => {
  // Track the last CAD reset time
  let lastResetCheck = null;

  // Check every 10 seconds if the CAD was reset
  setInterval(() => {
    fetch('php/get_last_reset.php')
      .then(res => res.text())
      .then(serverResetTime => {
        if (!lastResetCheck) {
          lastResetCheck = serverResetTime;
        } else if (serverResetTime !== lastResetCheck) {
          lastResetCheck = serverResetTime;
          console.log("CAD was reset — refreshing UI...");
          refreshData(); // You could also call window.location.reload();
        }
      });
  }, 10000); // check every 10 seconds

  // Auto-refresh CAD data every second
  setInterval(refreshData, 100000);

  // Load active events
  fetch('php/get_events.php')
    .then(res => res.json())
    .then(data => {
      data.forEach(event => {
        events[event.evNumber] = event;
        addEventToTable(event);
      });
    });

  // Load synced units
  fetch('php/api_units.php')
    .then(res => res.json())
    .then(data => {
      for (const callsign in units) {
        delete units[callsign];
      }
      document.getElementById("unitList").innerHTML = "";
      for (const callsign in data) {
        units[callsign] = data[callsign];
        renderUnit(units[callsign]);
      }
    });
};

const resetBtn = document.createElement("button");
resetBtn.textContent = "Reset CAD";
resetBtn.style.position = "fixed";
resetBtn.style.bottom = "10px";
resetBtn.style.right = "10px";
resetBtn.style.zIndex = "1000";
resetBtn.onclick = () => {
  if (confirm("Are you sure you want to reset the CAD? This will clear all saved data.")) {
    localStorage.clear();
    location.reload();
  }
};

document.body.appendChild(resetBtn);
